/* 

	One Wire Dallas Drivers, using a general IO Pin & UART...
		
	by Mauro Grassi, August 2010.

*/

#include "HardwareProfile.h"
#include "onewire.h"
#include "Delay.h"
#include "main.h"
#include "crc.h"
#include "uart.h"
#include "io.h"

#pragma udata onewiredata

#pragma code usercode

/* 
	The following 1 Wire Operations are Primitives

	Write 1 bit: 	Drive low, delay A, release, delay B
	Write 0 bit:	Drive low, delay C, release, delay D
	Read bit   :    Drive low, delay A, release, delay E, sample, delay F
	Reset pulse:    Delay G, drive low, delay H, release, delay I, sample (0=present slave), delay J
	
	Recommended delays:
	
	Standard or Legacy Timings in us
	
	A=6
	B=64
	C=60
	D=10
	E=9
	F=55
	G=0
	H=480
	I=70
	J=410
	
	Overdrive Timings in us
	
	A=1
	B=7.5
	C=7.5
	D=2.5
	E=1
	F=7
	G=2.5
	H=70
	I=8.5
	J=40

*/

unsigned char CRCOneWire(unsigned char* chr, unsigned int cnt)
{
	/* 
	
		works out the CRC for One Wire bus 
		The generator polynomial is X^8 + X^5 + X^4 + 1...
			
	*/	
	int i;
	unsigned char fb; 
	unsigned long a;
	unsigned int crc,Data; 
	crc=0; 
	for (a=0; a<cnt; a++)
	{ 
		Data=(unsigned int)chr[a]; 
		for (i=0;i<8;i++)
		{ 
			if(Data & 1)fb=(crc & 1) ^ 1; else fb=(crc & 1);
			crc=crc>>1;
			if(fb==1)
			{
				crc=crc ^ 0x8C;		/* 0x8C corresponds to the polynomial */
			}
			Data=Data>>1;
		}
	} 
	return((unsigned char)crc); 
}

void oneWireWaitus(unsigned int x)
{
	/* 	
		Only useful for delays greater than 8 us, 
		assumes instruction clock= 12MHz
		
		If x is less than or equal to 8us, it takes 4us to return...
		
	*/
	
	unsigned int c;

	if(x>=8)
	{
		/* stop the timer... */
		T0CONbits.TMR0ON=0;
		x=0x0017-x-x-x;
		TMR0H=(unsigned char)(x>>8);
		TMR0L=(unsigned char)(x);
		INTCONbits.TMR0IF=0;
		//INTCONbits.GIE=0;
		T0CON=0x81;
		while(INTCONbits.TMR0IF==0);
		T0CONbits.TMR0ON=0;
		//INTCONbits.GIE=1;
	}
}

void oneWireWaitms(unsigned int ms)
{
	while(ms--)
	{
		oneWireWaitus(1000);	
	}
}

unsigned char putResetPulse(HARDWARE_DESCRIPTOR* hd)
{
	/* 
		send a bus reset command, 
		holding the data line low for more than 480us will do it 
		returns 1 if a slave was detected, 0 otherwise...	
	*/
	unsigned char x;
	unsigned int  timeOut;
	unsigned char ch;
	unsigned char baudL;
	unsigned char baudH;
	
	if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
	{
		ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;
		
		if(hd->oneWirePort.mode & ONE_WIRE_OVERDRIVE)
		{
				/* delay G = 30 instructions at 12MIPS */
				INTCONbits.GIE=0;
				_asm
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
					nop
				_endasm
				
				openIOObject(ch, 0);
				/* drive low */
				oneWireWaitus(70);		/* delay H */
				openIOObject(ch, 1);			
				
				oneWireWaitus(9);		/* delay I */
				if(getIOObject(ch)==0)	/* sample */
				x=1; 
				else 
				x=0;	
				oneWireWaitus(40);		/* delay J */	
				INTCONbits.GIE=1;
				
		}
		else
		{	
		INTCONbits.GIE=0;
				
								/* delay G = 0 */
		openIOObject(ch, 0);	/* drive low */
		oneWireWaitus(480);		/* delay H */
		openIOObject(ch, 1);
		oneWireWaitus(70);		/* delay I */
		if(getIOObject(ch)==0)	/* sample */
		x=1; 
		else 
		x=0;
		oneWireWaitus(410);		/* delay J */	
		INTCONbits.GIE=1;
				
		}
	}
	else
	{
		/* save the values */
		baudH=SPBRGH2;
		baudL=SPBRG2;
		SPBRGH2=(unsigned char)ONE_WIRE_UART_BAUD_H_LOW_SPEED;
		SPBRG2=(unsigned char)ONE_WIRE_UART_BAUD_L_LOW_SPEED;
		PIR3bits.RC2IF=0;
		TXREG2=0xF0;
		timeOut=0;
		while((PIR3bits.RC2IF==0)&&(timeOut<ONE_WIRE_TIMEOUT))timeOut++;
		x=RCREG2;
		/* restore the values */
		SPBRGH2=baudH;
		SPBRG2=baudL;
		if((timeOut<ONE_WIRE_TIMEOUT)&&(x!=0xF0))x=1; else x=0;		
	}	
	return x;	
}

void putWriteSlot(HARDWARE_DESCRIPTOR* hd, unsigned char x)
{
	unsigned int  timeOut;
	unsigned char ch;
	
	timeOut=0;
	if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
	{	
		ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;
		if((x & 1)==1)
		{
			if(hd->oneWirePort.mode & ONE_WIRE_OVERDRIVE)
			{
			openIOObject(ch, 0);
			/* 12 nops = 1 us at 12MIPS */
			INTCONbits.GIE=0;
			_asm
				nop
				nop
				nop
				nop
				nop
				nop
				nop
				nop
				nop
				nop
				nop
				nop
			_endasm	
			openIOObject(ch, 1);
			oneWireWaitus(8);
			INTCONbits.GIE=1;		
			}
			else
			{
			INTCONbits.GIE=0;
			openIOObject(ch, 0);
			oneWireWaitus(6);
			openIOObject(ch, 1);
			oneWireWaitus(64);		
			INTCONbits.GIE=1;
			}
		}
		else
		{
			if(hd->oneWirePort.mode & ONE_WIRE_OVERDRIVE)
			{
				INTCONbits.GIE=0;
				openIOObject(ch, 0);
				oneWireWaitus(8);
				if((x & 2)==0)
				{
					openIOObject(ch, 1);
					/* 30 nops = 2.5us at 12 MIPS */
					_asm
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
						
						nop
						nop
						nop
					
						nop
						nop
						nop
						
						nop
						nop
						nop
					_endasm		
					INTCONbits.GIE=0;
				}
				else
				{
					setIOObject(ch, 1);
				}	
			}
			else
			{
				INTCONbits.GIE=0;
				openIOObject(ch, 0);
				oneWireWaitus(60);
				if((x & 2)==0)
				{
				openIOObject(ch, 1);
				oneWireWaitus(10);
				}
				else
				{
				setIOObject(ch, 1);
		 		}		
		 		INTCONbits.GIE=1;
			}
	}
	}
	else
	{
		ch=(hd->oneWireSerialPort.txrxpin>>4) & 0x0F;
		if((x & 1)==1)
		{
				PIR3bits.RC2IF=0;
				TXREG2=0xFF;
				while((PIR3bits.RC2IF==0)&&(timeOut<ONE_WIRE_TIMEOUT))timeOut++;
				x=RCREG2;
		}
		else
		{
				if(x & 2)
				{
				setIOObject(ch, 0);
				setRPNValue(ch, 0);
				oneWireWaitus(60);
				setIOObject(ch, 1);
				}
				else
				{
				PIR3bits.RC2IF=0;
				TXREG2=0x00;
				while((PIR3bits.RC2IF==0)&&(timeOut<ONE_WIRE_TIMEOUT))timeOut++;
				x=RCREG2;
				}
		}
	}
}	

unsigned char getReadBit(HARDWARE_DESCRIPTOR* hd)
{	
	/* read a bit from a slave */
	unsigned char x;
	unsigned int  timeOut;
	unsigned char ch;
	
	timeOut=0;
	
	if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
	{
		ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;

		if(hd->oneWirePort.mode & ONE_WIRE_OVERDRIVE)
		{
		
		openIOObject(ch, 0);
		
		/* 12 instructions = 1 us at 12 MIPS */
		INTCONbits.GIE=0;
		_asm
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
		_endasm
			
		openIOObject(ch, 1);
		
		/* 12 instructions = 1 us at 12 MIPS */
		_asm
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
		_endasm
		if(getIOObject(ch)==0)
		x=0; 
		else 
		x=1;
		oneWireWaitus(7);
		INTCONbits.GIE=1;
		}
		else
		{
		INTCONbits.GIE=0;
		openIOObject(ch, 0);
		oneWireWaitus(6);
		openIOObject(ch, 1);
		oneWireWaitus(9);
		if(getIOObject(ch)==0)
		x=0; 
		else 
		x=1;
		oneWireWaitus(55);
		INTCONbits.GIE=1;
		}
	}	
	else
	{
		PIR3bits.RC2IF=0;
		TXREG2=0xFF;
		while((PIR3bits.RC2IF==0)&&(timeOut<ONE_WIRE_TIMEOUT))timeOut++;
		x=RCREG2;
		if((timeOut>=ONE_WIRE_TIMEOUT)||(x==0xFF))x=1; else x=0;
	}
	return x;
}

char sendOneWireCommand(HARDWARE_DESCRIPTOR* hd, unsigned char cmd, unsigned int numBitsPacket, unsigned char* dataBuffer, unsigned char mode, unsigned int msPullUp)
{
	/* 

		Send an 8 bit command (cmd) and 
		
		Read or write up to numBitsPacket bits, with reply bits if any written to dataBuffer or read from there...
	  	also, returns the first char of the reply too...
	  	
		Note that data is sent/received LSB first...
		
		Mode indicates:
		
		READ/WRITE 			direction
		STRONG_PULLUP		uses strong pull up for parasite powered devices...

	*/
	
	unsigned char   i;
	unsigned char	j;
	unsigned char   tmp;
	unsigned char*  c;
	unsigned char   ch;
	
	#if(DEBUG_ONE_WIRE)
		putrs1USART((const MEM_MODEL rom char*)"Send CMD: ");
		disAUSART(cmd);
		putrs1USART((const MEM_MODEL rom char*)" Resp: ");
	#endif
	
	for(i=0; i<8; i++)
	{
		if((i==7)&&(mode & ONE_WIRE_STRONG_PULLUP))
		{
			if(cmd & 1)putWriteSlot(hd, 3); else putWriteSlot(hd, 2);		
		}
		else
		{
			if(cmd & 1)putWriteSlot(hd, 1); else putWriteSlot(hd, 0);
			cmd=cmd>>1;
		}	
	}
	
	if(mode & ONE_WIRE_STRONG_PULLUP)
	{
		/* enable strong pull up for this command */
		if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
		{		
		 ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;
		 oneWireWaitms(msPullUp);
		 openIOObject(ch, 1);
		 setIOObject(ch, 0);
		}
		else
		{
		 ch=(hd->oneWireSerialPort.txrxpin>>4) & 0x0F;
		 oneWireWaitms(msPullUp);
		 setRPNValue(ch, RPOR_VALUE_FOR_TX_UART);
		 tmp=RCREG2;		
		}
	}
	else
	if((numBitsPacket>0)&&(dataBuffer!=0))
	{
		if(mode & ONE_WIRE_READ_DIR)
		{
			i=0;
			c=dataBuffer;
			j=0;
			while(numBitsPacket>0)
			{
				tmp=tmp>>1;
				if(getReadBit(hd)==1)
				{
					tmp|=0x80;
				}
				else
				{
					tmp&=~0x80;
				}
				i++;
				if(i==8)
				{
					i=0;
					if(j<ONE_WIRE_RESPONSE_BUFFER_SIZE)
					{
						*dataBuffer++=tmp;
						j++;
					}
				}
				numBitsPacket--;
			}
			
			if(i!=0)
			{
				*dataBuffer++=tmp;
			}	
			return (*c);
			}
			else
			{
			
			i=0;
			c=dataBuffer;
			tmp=*dataBuffer++;
			while(numBitsPacket>0)
			{
				if(tmp & 1)
				{
					putWriteSlot(hd, 1);
				}
				else
				{
					putWriteSlot(hd, 0);
				}
				tmp=tmp>>1;
			i++;
			if(i==8)
			{
				i=0;
				tmp=*dataBuffer++;
			}
			numBitsPacket--;
		}
		return (*c);
		}	
	}
	else
	{
		return 0;
	}
}

#if 0
void showOneWireResponse(unsigned int numBits, unsigned char* pointer)
{
	while(numBits>0)
	{
		#if(DEBUG_ONE_WIRE)
		
		disAUSART(*pointer++);
		putrs1USART((const MEM_MODEL rom char*)".");
		
		#endif
		
		numBits-=8;
		
	}
}
#endif

void getOneWireState(HARDWARE_DESCRIPTOR* hd)
{
	unsigned char txch;
	unsigned char rxch;
	
	/* Fills in the appropriate UART information from the state */
	if((hd->oneWirePort.mode & NO_ONE_WIRE_MODE)||(hd->oneWirePort.mode & ONE_WIRE_USING_IO))
	{
			hd->oneWireSerialPort.mode=NO_UART_MODE;
			hd->oneWireSerialPort.txrxpin=NO_PIN;
			hd->oneWireSerialPort.baudRate=0;
	}
	else
	{
			txch=(hd->oneWirePort.mode & ONE_WIRE_PIN_MASK);
			if((txch>=0)&&(txch<=5))
			{
			
			}
			else
			{
				txch=0;
			}
			rxch=(txch+1) % 6;
			hd->oneWireSerialPort.txrxpin=(unsigned char)(txch<<4)+(unsigned char)(rxch);
			hd->oneWireSerialPort.mode=UART_OD | UART_NO_TX_INV | UART_NO_RX_INV;
			hd->oneWireSerialPort.baudRate=115200/BAUD_RATE_FACTOR;		
	}
}

void closeOneWireState(HARDWARE_DESCRIPTOR* hd)
{
	unsigned char ch;

	if(hd->oneWirePort.mode & NO_ONE_WIRE_MODE)
	{

	}
	else
	{	
		if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
		{
			ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;
			closeIOObject(ch);
		}
		else
		{
			closeUARTState(0, &hd->oneWireSerialPort);
		}
	}
}

void initOneWireState(HARDWARE_DESCRIPTOR* hd)
{
	unsigned int  i;
	unsigned char ch;
	
	/* Configure Timer 0 as counting 1:4 prescaler from (FOSC/4)=instruction clock or 12MHz */
		
	if(hd->oneWirePort.mode & NO_ONE_WIRE_MODE)
	{

	}
	else
	{
		if(hd->oneWirePort.mode & ONE_WIRE_USING_IO)
		{
			ch=hd->oneWirePort.mode & ONE_WIRE_PIN_MASK;
			setIOObject(ch, 0);
			openIOObject(ch, 1);
		}
		else
		{
			initUARTState(0, &hd->oneWireSerialPort);
		}
		}
}

void copyOneWireRomCode(HARDWARE_DESCRIPTOR* hd, unsigned char offset, unsigned char direction)
{
	/* copy one wire buffer to/from one wire rom code buffer */

	unsigned char rp;
	unsigned char op;
	unsigned char num;
	
	rp=0;
	op=offset;
	num=8;
	while(num--)
	{
		if(direction)
		{
			/* copy	rom code to one wire buffer */
			hd->oneWirePort.oneWireBuffer[op % ONE_WIRE_RESPONSE_BUFFER_SIZE]=hd->oneWirePort.romCode[rp++];
		}
		else
		{
			/* copy one wire buffer to rom code */
			hd->oneWirePort.romCode[rp++]=hd->oneWirePort.oneWireBuffer[op % ONE_WIRE_RESPONSE_BUFFER_SIZE];
		}
		op++;				
	}	
}				

#if 0
unsigned int readTemperature(ONE_WIRE_DEVICE* dev)
{
		unsigned int x;
		
		putResetPulse(dev);
		//putrs1USART((const MEM_MODEL rom char*)"\r\nSend Skip ROM Code\r\n");
		sendOneWireCommand(dev, 0xCC, 0, 0, ONE_WIRE_WRITE_DIR, 0);
		//putrs1USART((const MEM_MODEL rom char*)"\r\nSend Convert Temperature\r\n");
		sendOneWireCommand(dev, 0x44, 0, 0, ONE_WIRE_STRONG_PULLUP | ONE_WIRE_WRITE_DIR, 900);
		putResetPulse(dev);
		//putrs1USART((const MEM_MODEL rom char*)"\r\nSend Skip ROM Code\r\n");
		sendOneWireCommand(dev, 0xCC, 0, 0, ONE_WIRE_WRITE_DIR, 0);
		//putrs1USART((const MEM_MODEL rom char*)"\r\nSend Read ScratchPad: ");
		sendOneWireCommand(dev, 0xBE, 72, &oneWireResponseBuffer[0], ONE_WIRE_READ_DIR, 0);	
		//showOneWireResponse(72, &oneWireResponseBuffer[0]);		
		//putrs1USART((const MEM_MODEL rom char*)"\r\n");
		x=(unsigned int)oneWireResponseBuffer[0];
		x+=(((unsigned int)oneWireResponseBuffer[1])<<8);
		return x;
}
#endif
